/*
   This server expects, and echoes, two lines of input from
   the client. Then the client connection is closed by this
   server.

   Because this server is single threaded, it can only process
   input from one client at a time. While this server is waiting
   for a client's two lines of input, any new client that connects
   to this server is made to wait on this server's ServerSocket.

   To observe this server's listening socket queue up new clients,
   start a client, but don't type in any input to the client, and
   then start one or two more clients. Then go back to the first
   client and enter its input.
*/

import java.net.*;
import java.io.*;

public class EchoServer_v2
{
   public static final int SERVER_PORT = 5000; // Should be above 1023.

   public static void main (String[] args)
   {
      final int portNumber;
      if (args.length > 0)
      {
         portNumber = Integer.parseInt(args[0]);
      }
      else
      {
         portNumber = SERVER_PORT;
      }

      int clientCounter = 0;

      // Get this server's process id number (PID). This helps
      // to identify the server in TaskManager or TCPView.
      final ProcessHandle handle = ProcessHandle.current();
      final long pid = handle.pid();
      System.out.println("SERVER: Process ID number (PID): " + pid );

      // Get the name and IP address of the local host and
      // print them on the console for information purposes.
      try
      {
         final InetAddress address = InetAddress.getLocalHost();
         System.out.println("SERVER: Hostname: " + address.getCanonicalHostName() );
         System.out.println("SERVER: IP address: " +address.getHostAddress() );
         System.out.println("SERVER: Using port no. " + portNumber);
      }
      catch (UnknownHostException e)
      {
         System.out.println("Unable to determine this host's address.");
         System.out.println( e );
      }

      // Create the server's listening socket.
      ServerSocket serverSocket = null;
      try
      {
         serverSocket = new ServerSocket(portNumber); // Steps 1, 2.
         System.out.println("SERVER: online");
      }
      catch (IOException e)
      {
         System.out.println("SERVER: Error creating network connection.");
         e.printStackTrace();
         System.exit(-1);
      }

      while (true) // Serve multiple clients.
      {
         Socket socket = null;
         BufferedReader in = null;
         PrintWriter out = null;

         // Wait for an incoming client request.
         try
         {
            socket = serverSocket.accept(); // Steps 3, 6.

            // At this point, a client connection has been made.
            in = new BufferedReader(
                     new InputStreamReader(          // Step 8.
                          socket.getInputStream())); // Step 7.

            out = new PrintWriter(                  // Step 8.
                         socket.getOutputStream()); // Step 7.
         }
         catch (IOException e)
         {
            System.out.println("SERVER: Error connecting to client.");
            System.out.println( e );
         }

         ++clientCounter;
         // Get the client's IP address and port and log them to the console.
         final InetAddress clientIP = socket.getInetAddress();
         final int clientPort = socket.getPort();
         System.out.println("SERVER: Client " + clientCounter
                          + ": IP: " +  clientIP.getHostAddress()
                          + ", Port: " + clientPort);

         // Step 8.
         // Read the client's two requests and echo each one.
         try
         {
            // Wait for the client's first request.
            final String request1 = in.readLine();

            // Log the first request to the server's console.
            System.out.println("SERVER: Client " + clientCounter
                             + ": Message received: \"" + request1 + "\"");

            // Echo the first request back to the client.
            out.println("You are client " + clientCounter + ": " + request1);
            out.flush();      // Make sure that the response is sent.

            // Wait for the client's second request.
            final String request2 = in.readLine();

            // Log the second request to the server's console.
            System.out.println("SERVER: Client " + clientCounter
                             + ": Message received: \"" + request2 + "\"");

            // Echo the second request back to the client.
            out.println("You are client " + clientCounter + ": " + request2);
            out.flush(); // Make sure that the response is sent.

            // Step 9.
            // We are done with the client's requests.
            socket.close();

            System.out.println("SERVER: Client " + clientCounter
                             + ": Closed socket.");
         }
         catch (IOException e)
         {
            System.out.println("SERVER: Error communicating with client (Client no. " + clientCounter + ")");
            System.out.println( e );
         }
      }// Step 11 (wait for another client connection).
   }
}
