/*
   For each client, this server
      1) reads a sequence of positive integer values,
         one value per line of text,
         up until a negative integer value,
      2) sends back the sum of the positive integers as a text string,
      3) if not end-of-stream, go back to step 1,
      4) closes the connection to the client.
*/

import java.net.*;
import java.io.*;

public class AdditionServer_v3
{
   public static final int SERVER_PORT = 5000; // Should be above 1023.

   public static void main (String[] args)
   {
      final int portNumber;
      if (args.length > 0)
      {
         portNumber = Integer.parseInt(args[0]);
      }
      else
      {
         portNumber = SERVER_PORT;
      }

      int clientCounter = 0;

      // Get this server's process id number (PID). This helps
      // to identify the server in TaskManager or TCPView.
      final ProcessHandle handle = ProcessHandle.current();
      final long pid = handle.pid();
      System.out.println("SERVER: Process ID number (PID): " + pid );

      // Get the name and IP address of the local host and
      // print them on the console for information purposes.
      try
      {
         final InetAddress address = InetAddress.getLocalHost();
         System.out.println("SERVER: Hostname: " + address.getCanonicalHostName() );
         System.out.println("SERVER: IP address: " +address.getHostAddress() );
         System.out.println("SERVER: Using port no. " + portNumber);
      }
      catch (UnknownHostException e)
      {
         System.out.println("Unable to determine this host's address.");
         System.out.println( e );
      }

      // Create the server's listening socket.
      ServerSocket serverSocket = null;
      try
      {
         serverSocket = new ServerSocket(portNumber); // Steps 1, 2.
         System.out.println("SERVER online:");
      }
      catch (IOException e)
      {
         System.out.println("SERVER: Error creating network connection.");
         e.printStackTrace();
         System.exit(-1);
      }

      while (true) // Serve multiple clients.
      {
         Socket socket = null;
         BufferedReader in = null;
         PrintWriter out = null;

         // Wait for an incoming client request.
         try
         {
            socket = serverSocket.accept(); // Steps 3, 6.

            // At this point, a client connection has been made.
            in = new BufferedReader(
                     new InputStreamReader(          // Step 8.
                          socket.getInputStream())); // Step 7.

            out = new PrintWriter(                  // Step 8.
                         socket.getOutputStream()); // Step 7.
         }
         catch (IOException e)
         {
            System.out.println("SERVER: Error connecting to client.");
            System.out.println( e );
         }

         ++clientCounter;
         // Get the client's IP address and port and log them to the console.
         final InetAddress clientIP = socket.getInetAddress();
         final int clientPort = socket.getPort();
         System.out.println("SERVER: Client " + clientCounter
                          + ": IP: " +  clientIP.getHostAddress()
                          + ", Port: " + clientPort);


         // Step 8 - The application level protocol.
         // Read all of the client's requests.
         String request;
         try
         {
            while ( (request = in.readLine()) != null ) // While not end-of-stream.
            {
               int sum = 0;
               int n;
               while ( request != null  // While not end-of-stream.
                    && (n = Integer.parseInt(request.trim())) > 0 )
               {
                  sum += n;
                  request = in.readLine();
               }
               System.out.println("SERVER: Client " + clientCounter + ": Message received: sum = " + sum);
               out.println(sum); // Send the sum as a text string.
               out.flush();      // Now make sure that the response is sent.
            }
            socket.close();   // We are done with the client's request.
            System.out.println("SERVER: Client " + clientCounter + ": Closed socket.");
         }
         catch (IOException e)
         {
            System.out.println("SERVER: Error communicating with client (Client no. " + clientCounter + ")");
            System.out.println( e );
         }
      }
   }
}
