/*

*/

package TimeZoneWebApp_v3;

import java.text.DateFormat;
import java.util.Date;
import java.util.TimeZone;

import java.net.URL;
import java.net.URLConnection;
import java.net.MalformedURLException;

import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.Scanner;

/**
   This is an example of how we can use a web server and two
   web services to implement a very simple web application.

   This application expects the name of a time zone in its
   standard input stream and then uses a web service to look
   up that time zone's current time in some database. If the
   time zone is found in the database by the web service, then
   a web page containing the current time in that time zone is
   written to the standard output stream. If the time zone was
   not found in the database, then an error web page is written
   to the standard output stream.

   Dividing the "web application" between this program
   and a web service program lets this program worry
   only about the presentation of the results to the
   client. The web service program then worries about
   the logic of doing a database lookup of the city's
   time zone.

   To run this web application, copy this program's package folder
      TimeZoneWebApp_v3
   into the root folder of the web server. Then compile and run the
   web server that runs the web service program. Then access this URL.

         http://localhost:8080/TimeZoneWebApp_v3/
*/
public class TimeZoneApp
{
   private static final DateFormat timeFormatter = DateFormat.getTimeInstance();

   /**

   */
   public static void main(String[] args)
   {
      // Get the URL's query string from System.in.
      String query = null;
      try (final Scanner in = new Scanner(System.in))
      {
         query = in.nextLine();
      }

      // Parse it to find the city name.
      final int timeZoneIndex = query.indexOf("=") + 1;
      final String timeZoneId = query.substring(timeZoneIndex);

      // Send the time zone to the web service.
      String timeString = "null";
      try
      {
         final String resourceURL = "http://localhost:9090/TimeZoneWebService_v3/TimeZoneService.class?" + timeZoneId;
         final URL url = new URL(resourceURL);
         final URLConnection connection = url.openConnection();

         // Read one line from the server's response body.
         try (final var in = new BufferedReader(
                                new InputStreamReader(
                                   connection.getInputStream(), "latin1")))
         {
            timeString = in.readLine().trim();
         }
      }
      catch (MalformedURLException e)
      {
         System.err.println("=====> ERROR: TimeZoneApp: Bad URL.");
         System.err.println(e);
      }
      catch (IOException e)
      {
         System.err.println("=====> ERROR: TimeZoneApp: Cannot communicate with TimeZoneWebService");
         System.err.println(e);
      }

      // Log the response from the web service.
      System.err.println("=====> INFO: TimeZoneApp: time zone = " + timeZoneId + ", time = " + timeString);

      // Build the response body.
      final File file;
      if (! timeString.equals("null")) // if the city name was found
      {
         file = new File("TimeZoneWebApp_v3/next.html"); // result template page
      }
      else // if the city name was not found
      {
         file = new File("TimeZoneWebApp_v3/error.html"); // error template page
      }

      String body = "";
      try (final Scanner inFile = new Scanner(file))
      {
         while (inFile.hasNextLine()) // Read the whole html template page.
         {
             body += inFile.nextLine() + "\n";
         }
      }
      catch (FileNotFoundException e)
      {
         System.err.println("=====> ERROR: TimeZoneApp: Cannot find file: " + file);
         System.exit(-1);
      }

      // Replace the name template parameter with the name of the time zone.
      body = body.replaceFirst("###timeZone###", timeZoneId);

      // Replace the time template parameter with the time zone's time.
      body = body.replaceFirst("###timeZoneTime###", timeString);

      // Send the entity body.
      System.out.print(body);
      System.out.flush();
      System.out.close();
   }
}
